<?php
header('Content-Type: application/json; charset=utf-8');
require_once "config.php";
require_once "jdf.php";

try {
    session_start();

    $officeCode = $_SESSION['user_code'] ?? null;
    if (!$officeCode) {
        throw new Exception("کد مطب یافت نشد.");
    }

    $action = $_POST['action'] ?? null;
    if (!$action) {
        throw new Exception("هیچ عملیاتی ارسال نشده است.");
    }

    switch ($action) {
        case 'add_appointment':
            addAppointment($pdo, $officeCode);
            break;

        case 'load_appointments':
            loadAppointments($pdo, $officeCode);
            break;

        case 'delete_appointment':
            deleteAppointment($pdo, $officeCode);
            break;

        default:
            throw new Exception("عملیات نامعتبر: " . htmlspecialchars($action));
    }

} catch (Throwable $e) {
    echo json_encode([
        'success' => false,
        'error'   => $e->getMessage()
    ]);
}

// === FUNCTIONS ===

function loadAppointments($pdo, $officeCode) {
    $stmt = $pdo->prepare("
        SELECT *
        FROM appointments
        WHERE office_code = :office_code 
          AND status = 'ACTIVE'
    ");
    $stmt->execute([':office_code' => $officeCode]);
    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

    $events = [];
    foreach ($rows as $row) {
        $patientCode = $row['patient_code'];
        $service_category_code = $row['service_category_code'];
        $service_code = $row['service_code'];
        $dentist_code = $row['dentist_code'];
        $unit_code = $row['unit_code'];
        $notes = $row['notes'];
        // Get decrypted patient name
        $patientName = '';
        $pstmt = $pdo->prepare("SELECT patient_fullname_encrypted,iv,salt,enc_code FROM patients WHERE patient_code = ? LIMIT 1");
        $pstmt->execute([$row['patient_code']]);
        if ($pRow = $pstmt->fetch(PDO::FETCH_ASSOC)) {
            $patientName_enc = decryptData($pRow['patient_fullname_encrypted']);
            $patient_encrypted = [
                'patient_fullname_encrypted' => $patientName_enc
            ];
            $iv = $pRow['iv'];
            $salt = $pRow['salt'];
            $enc_code = $pRow['enc_code'];
            $tmp_patient = get_decrypted_data($patient_encrypted,$iv,$salt,$enc_code);
            $patientName = $tmp_patient['patient_fullname_encrypted'];
        }

        // Get service title
        $serviceTitle = '';
        $sstmt = $pdo->prepare("SELECT service_title FROM services WHERE service_code = ? LIMIT 1");
        $sstmt->execute([$row['service_code']]);
        if ($sRow = $sstmt->fetch(PDO::FETCH_ASSOC)) {
            $serviceTitle = $sRow['service_title'];
        }

        $year_start = (int) substr($row['start_datetime'], 0, 4);
        if($year_start<= 1420 && $year_start >= 1390){
            $start_date = jalaliDateTimeToGregorian($row['start_datetime']);
        }
        else{
            $start_date = $row['start_datetime'];
        }
        $length = $row['length'];
        $end_date = date('Y-m-d H:i:s', strtotime($start_date . ' +' . $length .' minutes'));
        // Create event object
        $events[] = [
            'id'    => $row['appointment_code'],
            'url'   => '',
            'title' => $patientName,
            'patientCode' => $patientCode,
            'start' => $start_date,
            'start_shamsi' => $row['start_datetime'],
            'end'   => $end_date,
            'length' => $length,
            'service_category_code' => $service_category_code,
            'service_code' => $service_code,
            'dentist_code' => $dentist_code,
            'unit_code' => $unit_code,
            'notes' => $notes,
            'allDay' => false,
            'extendedProps' => [
                'calendar' => 'Personal' // service_code as category
            ]
        ];
    }

    echo json_encode($events, JSON_UNESCAPED_UNICODE);
}
function addAppointment($pdo, $officeCode) {
    // Patient selection / addition logic
    $selectedPatientCode = trim($_POST['patient_select'] ?? '');
    $treatment_plan      = trim($_POST['treatment_plan'] ?? null);
    if ($selectedPatientCode !== '') {
        $patientCode = $selectedPatientCode;
    } else {
        // Add new patient (even if nationalid / phone empty)
        $patientName       = trim($_POST['patient_name'] ?? '');
        $patientNationalId = trim($_POST['patient_nationalid'] ?? '');
        $patientPhone      = trim($_POST['patient_phone'] ?? '');
        // Split to firstname / lastname if space found
        $nameParts = preg_split('/\s+/', $patientName, 2);
        $firstName = $nameParts[0] ?? '';
        $lastName  = $nameParts[1] ?? '';

        $payload = [];
        $payload['patient_name'] = trim($_POST['patient_name'] ?? '');
        $payload['patient_nationalid']  = trim($_POST['patient_nationalid'] ?? '');
        $payload['patient_phone']     = trim($_POST['patient_phone'] ?? '');
        $payload['firstname']     = trim($firstName ?? '');
        $payload['lastname']     = trim($lastName ?? '');
        $enc = get_encrypted_data(json_encode($payload));
        $enc_data = $enc['encrypted_data'];
        $iv = $enc['iv'];
        $salt = $enc['salt'];
        $enc_code = $enc['code'];
        $patientName_enc       = trim($enc_data['patient_name'] ?? '');
        $patientNationalId_enc  = trim($enc_data['patient_nationalid'] ?? '');
        $patientPhone_enc       = trim($enc_data['patient_phone'] ?? '');
        $firstname_enc       = trim($enc_data['firstname'] ?? '');
        $lastname_enc       = trim($enc_data['lastname'] ?? '');
        $stmt = $pdo->prepare("
        SELECT case_number_encrypted 
        FROM patients 
        WHERE office_code = :office
    ");
        $stmt->execute([':office' => $officeCode]);
        $maxCase = 0;
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            $decrypted = decryptData($row['case_number_encrypted']); // same as encryptData's reverse
            if (ctype_digit($decrypted) && (int)$decrypted > $maxCase) {
                $maxCase = (int)$decrypted;
            }
        }
        $casenNumber = $maxCase + 1;

        if ($patientName === '') {
            throw new Exception('نام بیمار الزامی است.');
        }

        $patientCode = uniqid("PAT-") . '-' . substr(base64_encode(hash('sha256', $officeCode)), 0, 20);

        $stmt = $pdo->prepare("
            INSERT INTO patients (
                patient_code,
                patient_firstname_encrypted, patient_firstname_hash,
                patient_lastname_encrypted, patient_lastname_hash,
                patient_fullname_encrypted, patient_fullname_hash,
                patient_nationalid_encrypted, patient_nationalid_hash,
                patient_phone_encrypted, patient_phone_hash,
                office_code,case_number_encrypted,case_number_hash,iv,salt,enc_code,
                status
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?,?,?, ?, ?, ?, 'ACTIVE')
        ");

        $stmt->execute([
            $patientCode,
            encryptData($firstname_enc),          hashData($firstName),
            encryptData($lastname_enc),           hashData($lastName),
            encryptData($patientName_enc),        hashData($patientName),
            encryptData($patientNationalId_enc),  hashData($patientNationalId),
            encryptData($patientPhone_enc),       hashData($patientPhone),
            $officeCode, encryptData($casenNumber), hashData($casenNumber),$iv,$salt,$enc_code
        ]);
    }

    // Appointment fields
    $serviceCategoryCode = trim($_POST['service_category_code'] ?? '');
    $serviceCode = trim($_POST['service_code'] ?? '');
    $dentistCode = trim($_POST['dentist_code'] ?? '');
    $unitCode    = trim($_POST['unit_code'] ?? '');
    $start       = trim($_POST['eventStartDate'] ?? '');
    $length         = trim($_POST['eventLength'] ?? '');
    $notes       = trim($_POST['eventDescription'] ?? '');
    $hour  = trim($_POST['eventStartTime'] ?? '00:00');

// Combine into a single string
    if ($start === '') {
        throw new Exception('تاریخ شروع الزامی است.');
    }
    if ($hour === '') {
        throw new Exception('ساعت نوبت الزامی است.');
    }
    $appointment_start = $start . ' ' . $hour . ':00';

    $appointmentCode = uniqid("APP-") . '-'  . substr(base64_encode(hash('sha256', $officeCode, true)), 0, 20);;

    $stmt = $pdo->prepare("
        INSERT INTO appointments (
            appointment_code,
            treatment_plan_code,
            patient_code,
            service_code,
            service_category_code,
            dentist_code,
            unit_code,
            start_datetime,
            length,
            notes,
            office_code,
            status
        ) VALUES (?, ?, ?, ?, ?, ?,?, ?, ?, ?, ?, 'ACTIVE')
    ");

    $stmt->execute([
        $appointmentCode,
        $treatment_plan,
        $patientCode,
        $serviceCode,
        $serviceCategoryCode,
        $dentistCode,
        $unitCode,
        $appointment_start,
        $length,
        $notes,
        $officeCode
    ]);
// === DEBUG: Save all $_POST to a text log ===
    $logFile = __DIR__ . '/post_debug.log'; // saved in functions/ folder
    $logData = date('Y-m-d H:i:s') . ' | ' . print_r($_POST, true) . PHP_EOL;
    file_put_contents($logFile, $logData, FILE_APPEND | LOCK_EX);

    echo json_encode([
        'success' => true,
        'message' => 'نوبت با موفقیت ثبت شد.',
        'appointment_code' => $appointmentCode
    ]);
}

function deleteAppointment($pdo, $officeCode) {
    $code = trim($_POST['code'] ?? '');
    if ($code === '') {
        echo json_encode(['success' => false, 'error' => 'شناسه نوبت ارسال نشده است.']);
        return;
    }

    $stmt = $pdo->prepare("
        UPDATE appointments
        SET status = 'DELETED'
        WHERE appointment_code = :code
          AND office_code = :office_code
    ");
    $stmt->execute([
        ':code'        => $code,
        ':office_code' => $officeCode
    ]);

    if ($stmt->rowCount()) {
        echo json_encode(['success' => true, 'message' => 'نوبت با موفقیت حذف شد.']);
    } else {
        echo json_encode(['success' => false, 'error' => 'نوبتی برای حذف یافت نشد.']);
    }
}
