<?php
header('Content-Type: application/json; charset=utf-8');
require_once "config.php";

try {
    session_start();

    $officeCode = $_SESSION['user_code'] ?? null;
    if (!$officeCode) {
        throw new Exception("No office code found in session.");
    }

    $action = $_POST['action'] ?? null;
    if (!$action) {
        throw new Exception("No action provided.");
    }

    switch ($action) {
        case 'add_dentist':
            addDentist($pdo, $officeCode);
            break;

        case 'delete_dentist':
            deleteDentist($pdo, $officeCode);
            exit;

        case 'load_dentists':
            loadDentists($pdo, $officeCode);
            break;

        case 'get_dentist':
            editDentist($pdo, $officeCode);
            exit;

        case 'update_dentist':
            updateDentist($pdo, $officeCode);
            exit;

        default:
            throw new Exception("Invalid action: " . htmlspecialchars($action));
    }

} catch (Throwable $e) {
    echo json_encode([
        'success' => false,
        'error'   => $e->getMessage()
    ]);
}

// === FUNCTIONS ===

function loadDentists($pdo, $officeCode) {
    $stmt = $pdo->prepare("
        SELECT
            id,
            dentist_code,
            dentist_firstname_encrypted,
            dentist_lastname_encrypted,
            dentist_speciality_encrypted,
            dentist_nationalid_encrypted,
            dentist_nezamid_encrypted,iv,salt,enc_code,cut
        FROM dentists
        WHERE status = 'ACTIVE'
          AND office_code = :office_code
    ");
    $stmt->execute([':office_code' => $officeCode]);

    $dentists = [];
    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
        $dentist_encrypted = [
            'dentist_firstname_encrypted' => decryptData($row['dentist_firstname_encrypted']),
            'dentist_lastname_encrypted'  => decryptData($row['dentist_lastname_encrypted']),
            'dentist_speciality_encrypted'=> decryptData($row['dentist_speciality_encrypted']),
            'dentist_nationalid_encrypted'=> decryptData($row['dentist_nationalid_encrypted']),
            'dentist_nezamid_encrypted'   => decryptData($row['dentist_nezamid_encrypted'])
        ];
        $iv = $row['iv'];
        $salt = $row['salt'];
        $enc_code = $row['enc_code'];
        $tmp_dentist = get_decrypted_data($dentist_encrypted,$iv,$salt,$enc_code);
        $dentists[] = [
            'firstname' => $tmp_dentist['dentist_firstname_encrypted'],
            'lastname'  => $tmp_dentist['dentist_lastname_encrypted'],
            'speciality'=> $tmp_dentist['dentist_speciality_encrypted'],
            'nationalid'=> $tmp_dentist['dentist_nationalid_encrypted'],
            'nezamid'   => $tmp_dentist['dentist_nezamid_encrypted'],
            'cut'       => $row['cut'],
            'status'    => '<span class="badge bg-label-success">فعال</span>',
            'actions'   => '<button class="btn btn-sm btn-warning edit-dentist" data-code="'.$row['dentist_code'].'">ویرایش</button>
                             <button class="btn btn-sm btn-danger delete-dentist" data-code="'.$row['dentist_code'].'">حذف</button>'
        ];
    }
    echo json_encode(['data' => $dentists]);
}

function addDentist($pdo, $officeCode) {
    $payload = [];
    $payload['add-dentist-firstname'] = trim($_POST['add-dentist-firstname'] ?? '');
    $payload['add-dentist-lastname']  = trim($_POST['add-dentist-lastname'] ?? '');
    $payload['add-dentist-fullname']  = trim($_POST['add-dentist-firstname'] . ' ' . $_POST['add-dentist-lastname'] ?? '');
    $payload['add-dentist-phone']     = trim($_POST['add-dentist-phone'] ?? '');
    $payload['add-dentist-speciality']= trim($_POST['add-dentist-speciality'] ?? '');
    $payload['add-dentist-nationalid']= trim($_POST['add-dentist-nationalid'] ?? '');
    $payload['add-dentist-nezamid']   = trim($_POST['add-dentist-nezamid'] ?? '');
    $payload['add-dentist-cut']   = trim($_POST['add-dentist-cut'] ?? '');

    $firstName = trim($_POST['add-dentist-firstname'] ?? '');
    $lastName  = trim($_POST['add-dentist-lastname'] ?? '');
    $phone     = trim($_POST['add-dentist-phone'] ?? '');
    $speciality= trim($_POST['add-dentist-speciality'] ?? '');
    $nationalId= trim($_POST['add-dentist-nationalid'] ?? '');
    $nezamId   = trim($_POST['add-dentist-nezamid'] ?? '');
    $cut   = trim($_POST['add-dentist-cut'] ?? '');

    $enc = get_encrypted_data(json_encode($payload));
    $enc_data = $enc['encrypted_data'];
    //file_put_contents(__DIR__ . '/test2.txt', $enc['add-dentist-firstname']);
    $firstName_enc = trim($enc_data['add-dentist-firstname'] ?? '');
    $lastName_enc  = trim($enc_data['add-dentist-lastname'] ?? '');
    $fullName_enc  = trim($enc_data['add-dentist-fullname'] ?? '');
    $phone_enc     = trim($enc_data['add-dentist-phone'] ?? '');
    $speciality_enc= trim($enc_data['add-dentist-speciality'] ?? '');
    $nationalId_enc= trim($enc_data['add-dentist-nationalid'] ?? '');
    $nezamId_enc   = trim($enc_data['add-dentist-nezamid'] ?? '');
    $iv = $enc['iv'];
    $salt = $enc['salt'];
    $enc_code = $enc['code'];
    if (!$firstName || !$lastName) {
        throw new Exception("نام و نام خانوادگی را وارد نمایید.");
    }

    $fullName   = $firstName . ' ' . $lastName;
    $dentistCode= uniqid("DNT-") . '-' . substr(base64_encode(hash('sha256', $officeCode)), 0, 20);

    $stmt = $pdo->prepare("
        INSERT INTO dentists (
            dentist_code,
            dentist_firstname_encrypted, dentist_firstname_hash,
            dentist_lastname_encrypted, dentist_lastname_hash,
            dentist_fullname_encrypted, dentist_fullname_hash,
            dentist_nationalid_encrypted, dentist_nationalid_hash,
            dentist_phone_encrypted, dentist_phone_hash,
            dentist_nezamid_encrypted, dentist_nezamid_hash,
            dentist_speciality_encrypted, dentist_speciality_hash,cut,
            office_code,iv,salt,enc_code
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
    ");

    $stmt->execute([
        $dentistCode,
        encryptData($firstName_enc), hashData($firstName),
        encryptData($lastName_enc), hashData($lastName),
        encryptData($fullName_enc), hashData($fullName),
        encryptData($nationalId_enc), hashData($nationalId),
        encryptData($phone_enc), hashData($phone),
        encryptData($nezamId_enc), hashData($nezamId),
        encryptData($speciality_enc), hashData($speciality),$cut,
        $officeCode,$iv,$salt,$enc_code
    ]);

    echo json_encode([
        'success' => true,
        'message' => 'دندانپزشک با موفقیت افزوده شد.'
    ]);
}

function deleteDentist($pdo, $officeCode){
    if (empty($_POST['code'])) {
        echo json_encode(['success' => false, 'error' => 'شناسه دندانپزشک ارسال نشده است']);
        exit;
    }
    $code = trim($_POST['code']);
    try {
        $stmt = $pdo->prepare("
            UPDATE dentists
            SET status = 'DELETED'
            WHERE dentist_code = :code
              AND office_code = :office_code
        ");
        $stmt->execute([
            ':code'        => $code,
            ':office_code' => $officeCode
        ]);

        if ($stmt->rowCount()) {
            echo json_encode(['success' => true, 'message' => 'دندانپزشک با موفقیت حذف شد.']);
        } else {
            echo json_encode(['success' => false, 'error' => 'دندانپزشکی برای حذف یافت نشد.']);
        }
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'error' => 'خطای پایگاه داده: ' . $e->getMessage()]);
    }
}

function editDentist($pdo, $officeCode){
    if (!isset($_POST['code'])) {
        echo json_encode([
            'success' => false,
            'error'   => 'شناسه معتبر نمی‌باشد'
        ]);
        exit;
    }

    $code = trim($_POST['code']);

    try {
        $stmt = $pdo->prepare("
            SELECT
                dentist_code,
                dentist_firstname_encrypted,
                dentist_lastname_encrypted,
                dentist_speciality_encrypted,
                dentist_nationalid_encrypted,
                dentist_nezamid_encrypted,cut,iv,salt,enc_code
            FROM dentists
            WHERE dentist_code = :code
              AND office_code = :office_code
              AND status != 'DELETED'
            LIMIT 1
        ");
        $stmt->execute([
            ':code'        => $code,
            ':office_code' => $officeCode
        ]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($row) {
            $dentist_encrypted = [
                'dentist_firstname_encrypted' => decryptData($row['dentist_firstname_encrypted']),
                'dentist_lastname_encrypted'  => decryptData($row['dentist_lastname_encrypted']),
                'dentist_speciality_encrypted'=> decryptData($row['dentist_speciality_encrypted']),
                'dentist_nationalid_encrypted'=> decryptData($row['dentist_nationalid_encrypted']),
                'dentist_nezamid_encrypted'   => decryptData($row['dentist_nezamid_encrypted'])
            ];
            $iv = $row['iv'];
            $salt = $row['salt'];
            $enc_code = $row['enc_code'];
            $tmp_dentist = get_decrypted_data($dentist_encrypted,$iv,$salt,$enc_code);
            echo json_encode([
                'success' => true,
                'dentist' => [
                    'id'         => $row['dentist_code'],
                    'firstname'  => $tmp_dentist['dentist_firstname_encrypted'],
                    'lastname'   => $tmp_dentist['dentist_lastname_encrypted'],
                    'speciality' => $tmp_dentist['dentist_speciality_encrypted'],
                    'nationalid' => $tmp_dentist['dentist_nationalid_encrypted'],
                    'nezamid'    => $tmp_dentist['dentist_nezamid_encrypted'],
                    'cut'        => $row['cut']
                ]
            ]);
        } else {
            echo json_encode([
                'success' => false,
                'error'   => 'دندانپزشک یافت نشد'
            ]);
        }

    } catch (Exception $e) {
        echo json_encode([
            'success' => false,
            'error'   => 'خطای داخلی سرور'
        ]);
    }
}

function updateDentist($pdo, $officeCode){
    if (empty($_POST['code'])) {
        echo json_encode([
            'success' => false,
            'error'   => 'کد دندانپزشک ارسال نشده است.'
        ]);
        exit;
    }
    $payload = [];
    $payload['firstname'] = trim($_POST['firstname'] ?? '');
    $payload['lastname']  = trim($_POST['lastname'] ?? '');
    $payload['fullname']  = trim($_POST['firstname'] . ' ' . $_POST['lastname'] ?? '');
    $payload['speciality']=  trim($_POST['speciality'] ?? '');
    $payload['nationalid']= trim($_POST['nationalid'] ?? '');
    $payload['nezamid']   = trim($_POST['nezamid'] ?? '');

    $enc = get_encrypted_data(json_encode($payload));
    $enc_data = $enc['encrypted_data'];

    $dentistCode = trim($_POST['code']);
    $firstname   = trim($_POST['firstname'] ?? '');
    $lastname    = trim($_POST['lastname'] ?? '');
    $speciality  = trim($_POST['speciality'] ?? '');
    $nationalid  = trim($_POST['nationalid'] ?? '');
    $nezamid     = trim($_POST['nezamid'] ?? '');
    $cut     = trim($_POST['cut'] ?? '');

    $firstname_enc  = trim($enc_data['firstname'] ?? '');
    $lastname_enc    = trim($enc_data['lastname'] ?? '');
    $fullname_enc    = trim($enc_data['fullname'] ?? '');
    $speciality_enc  = trim($enc_data['speciality'] ?? '');
    $nationalid_enc  = trim($enc_data['nationalid'] ?? '');
    $nezamid_enc     = trim($enc_data['nezamid'] ?? '');
    $iv = $enc['iv'];
    $salt = $enc['salt'];
    $enc_code = $enc['code'];
    if ($firstname === '' || $lastname === '' || $speciality === '' || $nationalid === '' || $nezamid === '') {
        echo json_encode([
            'success' => false,
            'error'   => 'تمام فیلدها الزامی هستند.'
        ]);
        exit;
    }

    try {
        $stmt = $pdo->prepare("
            UPDATE dentists
            SET
                dentist_firstname_encrypted   = :firstname,
                dentist_lastname_encrypted    = :lastname,
                dentist_fullname_encrypted    = :fullname,
                dentist_speciality_encrypted  = :speciality,
                dentist_nationalid_encrypted  = :nationalid,
                dentist_nezamid_encrypted     = :nezamid,
                cut                           = :cut,
                iv = :iv,salt = :salt,enc_code = :enc_code
            WHERE dentist_code = :code
              AND office_code  = :office_code
              AND status != 'DELETED'
        ");
        $updated = $stmt->execute([
            ':firstname'   => encryptData($firstname_enc),
            ':lastname'    => encryptData($lastname_enc),
            ':fullname'    => encryptData($fullname_enc),
            ':speciality'  => encryptData($speciality_enc),
            ':nationalid'  => encryptData($nationalid_enc),
            ':nezamid'     => encryptData($nezamid_enc),
            ':cut'         => $cut,
            ':iv'          => $iv,
            ':salt'        => $salt,
            ':enc_code'    => $enc_code,
            ':code'        => $dentistCode,
            ':office_code' => $officeCode
        ]);

        if ($updated && $stmt->rowCount() > 0) {
            echo json_encode([
                'success' => true,
                'message' => 'اطلاعات دندانپزشک با موفقیت بروزرسانی شد.'
            ]);
        } else {
            echo json_encode([
                'success' => false,
                'error'   => 'دندانپزشک یافت نشد یا تغییری در اطلاعات ایجاد نشد.'
            ]);
        }

    } catch (Exception $e) {
        echo json_encode([
            'success' => false,
            'error'   => 'خطای داخلی سرور.'
        ]);
    }
}
