<?php
header('Content-Type: application/json; charset=utf-8');
require_once "config.php";
try {
    session_start();
    $action = $_POST['action'] ?? null;
    if (!$action) throw new Exception("No action provided.");

    switch ($action) {
        case 'load_units':
            loadUnits($pdo);
            break;
        case 'add_unit':
            addUnit($pdo);
            break;
        case 'delete_unit':
            deleteUnit($pdo);
            break;
        default:
            throw new Exception("Invalid action: " . htmlspecialchars($action));
    }
} catch (Throwable $e) {
    echo json_encode(['success' => false, 'error' => $e->getMessage()]);
}

// === FUNCTIONS ===

function loadUnits($pdo) {
    $officeCode =  $_SESSION['user_code'] ?? null;
    $stmt = $pdo->prepare("SELECT * FROM dental_chairs WHERE office_code = ? AND status != 'deleted'");
    $stmt->execute([$officeCode]);
    $units = [];
    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
        // Prepare dentists badges from JSON
        $dentistBadges = '';
        if (!empty($row['assigned_dentists']) && json_validate($row['assigned_dentists'])) {
            $ids = json_decode($row['assigned_dentists'], true);
            if (is_array($ids) && count($ids)) {
                // Load those dentists from DB
                $in  = str_repeat('?,', count($ids) - 1) . '?';
                $q   = $pdo->prepare("SELECT dentist_firstname_encrypted, dentist_lastname_encrypted,iv,salt,enc_code FROM dentists WHERE dentist_code IN ($in)");
                $q->execute($ids);
                while ($d = $q->fetch(PDO::FETCH_ASSOC)) {
                    $dentist_encrypted[] = [
                        'dentist_firstname_encrypted' => decryptData($d['dentist_firstname_encrypted']),
                        'dentist_lastname_encrypted'  => decryptData($d['dentist_lastname_encrypted'])
                    ];
                    $iv = $d['iv'];
                    $salt = $d['salt'];
                    $enc_code = $d['enc_code'];
                    $tmp_dentist = get_decrypted_data($dentist_encrypted,$iv,$salt,$enc_code);
                    $fname = $tmp_dentist['dentist_firstname_encrypted'];
                    $lname = $tmp_dentist['dentist_lastname_encrypted'];
                    $dentistBadges .= '<span class="badge bg-label-primary me-1">'.$fname.' '.$lname.'</span>';
                }
            }
        }
        // Status badge
        $statusBadge = match($row['status']) {
            'available' => '<span class="badge bg-label-success">آزاد</span>',
            'in_use' => '<span class="badge bg-label-warning">در حال استفاده</span>',
            'out_of_order' => '<span class="badge bg-label-danger">خراب</span>',
            default => '<span class="badge bg-label-secondary">نامشخص</span>',
        };
        $units[] = [
            'code'     => $row['unit_code'],
            'name'     => $row['unit_name'],
            'location' => $row['location'],
            'status'   => $statusBadge,
            'dentists' => $dentistBadges,
            'actions'  => '<button class="btn btn-sm btn-danger delete-unit" data-code="'.$row['unit_code'].'">حذف</button>'
        ];
    }
    echo json_encode(['data' => $units]);
}

function addUnit($pdo) {
    $code     = uniqid("UNT-") . '-' . substr(base64_encode(hash('sha256', $officeCode)), 0, 20);
    $name     = trim($_POST['unit-name'] ?? '');
    $location = trim($_POST['unit-location'] ?? '');
    $status   = trim($_POST['unit-status'] ?? 'available');
    $dentists = $_POST['unit-dentists'] ?? [];
    $officeCode = $_SESSION['user_code'] ?? null;
    if (!$name) throw new Exception("نام یونیت الزامی است.");
    if (!is_array($dentists)) $dentists = [];

    $dentistsJson = json_encode(array_map('strval', $dentists), JSON_UNESCAPED_UNICODE);

    $stmt = $pdo->prepare("INSERT INTO dental_chairs (unit_code, unit_name, location, status, assigned_dentists,office_code) VALUES (?, ?, ?, ?, ?, ?)");
    $stmt->execute([$code, $name, $location, $status, $dentistsJson,$officeCode]);

    echo json_encode(['success' => true, 'message' => 'یونیت با موفقیت افزوده شد.']);
}

function deleteUnit($pdo) {
    if (empty($_POST['code'])) throw new Exception('شناسه یونیت ارسال نشده است');
    $code = trim($_POST['code']);
    $stmt = $pdo->prepare("UPDATE dental_chairs SET status = 'deleted' WHERE unit_code = :code");
    $stmt->execute([':code' => $code]);
    if ($stmt->rowCount()) {
        echo json_encode(['success' => true, 'message' => 'یونیت با موفقیت حذف شد.']);
    } else {
        echo json_encode(['success' => false, 'error' => 'یونیتی برای حذف یافت نشد.']);
    }
}

// Helper to check if JSON is valid
function json_validate($string) {
    if (!is_string($string)) return false;
    json_decode($string);
    return json_last_error() === JSON_ERROR_NONE;
}
